% Model of a nonisothermal batch reactor - Alkylation
%                                          Two reactions

% Main file

% For more information, see:

% A. De Visscher 
% Lecture Notes in Chemical Engineering Kinetics and Chemical Reactor Design
% Createspace (2013) 

% (C) Alex De Visscher (2013)

% ---------------------------

% data

clear all

tspan = 0:2:200;     % We want output every 5 s up to 500 s
cC40 = 0;            % Initial butene concentration (kmol/m3)
ciC40 = 9.48;        % Initial isobutane concentration (kmol/m3)
ciC80 = 0;           % Initial isooctane concentration (kmol/m3)
ciC120 = 0;          % Initial isododecane concentration (kmol/m3)

% k1                 % Rate constant (m3/kg/s)
A = 265;
EA = 10000;

% K                  % Rate constant (-)
B = 30;
H = 15000;

Vinit = 20;          % Initial reactor volume (m3)
m = 5;               % Catalyst mass (kg)
Ac = 100;            % Area of heating/cooling coil (m2)
U = 0.500;           % Heat transfer coefficient (kJ/m2 K)
Tc = 373.15;         % Temperature of heat transfer medium (K)
Tin = 298.15;        % Temperature of influent (K)
ws = 1000;           % shaft work (J)

DH0 = -65100;        % Enthalpy of reaction at 25 C
CpiC4 = 129;         % Heat capacity of isobutane (kJ/kmol K)
CpC4 = 124;          % Heat capacity of butene (kJ/kmol K)
CpiC8 = 247;         % Heat capacity of isooctane (kJ/kmol K)
DCp = CpiC8 - CpiC4 - CpC4;  % Heat capacity of reaction
CpiC12 = CpiC8 + CpC4 + DCp; % Heat capacity isododecane

n0(1) = cC40*Vinit;  % Initial number of kmol of C4
n0(2) = ciC40*Vinit; % Initial number of kmol of i-C4
n0(3) = ciC80*Vinit; % Initial number of kmol of i-C8
n0(4) = ciC120*Vinit;% Initial number of kmol of i-C12
n0(5) = 373.15;      % Initial temperature (K)
n0(6) = Vinit;       % Initial volume (m3)

nin(1) = 0.125;       % Incoming flow rate of C4 (kmol/s)
nin(2) = 0.650;       % Incoming flow rate of i-C4
nin(3) = 0;          % Incoming flow rate of i-C8
nin(4) = 0;          % Incoming flow rate of i-C12

M(1) = 56;           % Molar mass of C4 (kg/kmol)
M(2) = 58;           % Molar mass of i-C4
M(3) = 114;          % Molar mass of i-C8
M(4) = 170;          % Molar mass of i-C12
rho = 550;           % Liquid density (kg/m3)

% Call of differential equation solver

options = odeset('RelTol', 1e-6, 'AbsTol', 1e-8, 'InitialStep', 0.01);
[T,N] = ode45(@f,tspan,n0,options,A,EA,B,H,m,Ac,U,Tc,Tin,ws,DH0,CpiC4,CpC4,CpiC8,CpiC12,DCp,nin,rho,M);

% Output

figure(1)
plot(T,N(:,1:4))
xlabel('Time (s)');
ylabel('Amount (kmol)');
figure(2)
plot(T,N(:,5))
xlabel('Time (s)');
ylabel('Temperature (K)');
V = N(:,6);
C = N(:,1:4);
for j = 1:4
    C(:,j) = C(:,j)./V;
end
figure(3)
plot(T,N(:,6))
xlabel('Time(s)');
ylabel('Volume (m3)');
figure(4)
plot(T,C)
xlabel('Time (s)');
ylabel('Concentration (kmol/m3)');
T
C
[s1,s2] = size(T);
conversion = (N(1,1) + T(s1)*nin(1) - N(s1,1))/(N(1,1) + T(s1)*nin(1))
conv_iC8 = N(s1,3)/(N(1,1) + T(s1)*nin(1))

